<?php

/**
 * Elementor Widget
 * @package Vibenews
 * @since 1.0.0
 */


if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly.

class vibenews_post_block_item_three extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve Elementor widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'posts-block-item-three';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Elementor widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Post Block Item 3', 'vibenews-extra' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Elementor widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'fas fa-th-large'; 
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Elementor widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'vibenews_widgets' ];
	}

	/**
	 * Register Elementor widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	 
	
	protected function register_controls() {
		$this->layout_options();		
		$this->post_query_options();	
		$this->meta_options();	
		$this->design_options();
	}
	
	/**
    * Layout Options
    */
    private function layout_options() {
	
	
		$this->start_controls_section(
            'layout_option',
            [
                'label' => __( 'Layout Options', 'vibenews-extra' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
		
		
		$this->add_control(
			'post_block_title',
			[
				'label' => __( 'Section Title', 'vibenews-extra' ),
				'type' => \Elementor\Controls_Manager::TEXTAREA,
				'default' => __( 'Whats Happening', 'vibenews-extra' ),
				'placeholder' => __( 'Type Your title here', 'vibenews-extra' ),
			]
		);
		
		$this->add_control(
            'display_three_top_title',
            [
                'label' => esc_html__('Display Top Heading', 'vibenews-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'vibenews-extra'),
                'label_off' => esc_html__('No', 'vibenews-extra'),
                'default' => 'yes',
            ]
        );
		
		$this->add_responsive_control(
			'grid_img_width',
			[
				'label' =>esc_html__( 'Set Post Image Width', 'kristo-extra' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
					],
				],
				'devices' => [ 'desktop', 'tablet', 'mobile' ],
				'desktop_default' => [
					'size' => 490,
					'unit' => 'px',
				],
				'tablet_default' => [
					'size' => 490,
					'unit' => 'px',
				],
				'mobile_default' => [
					'size' => 490,
					'unit' => 'px',
				],
				
				'default'  => [
					'unit' => 'px',
					'size' => 490,
				],
				
				'selectors' => [
				
					'{{WRAPPER}} .post-block-item-three .news-post-grid-thumbnail' => 'min-width: {{SIZE}}{{UNIT}};',
				],
			]
		);
		
		$this->add_responsive_control(
			'grid_img_height',
			[
				'label' =>esc_html__( 'Set Post Image Height', 'vibenews-extra' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
					],
				],
				'devices' => [ 'desktop', 'tablet', 'mobile' ],
				'desktop_default' => [
					'size' => 345,
					'unit' => 'px',
				],
				'tablet_default' => [
					'size' => 345,
					'unit' => 'px',
				],
				'mobile_default' => [
					'size' => 345,
					'unit' => 'px',
				],
				
				'default'  => [
					'unit' => 'px',
					'size' => 345,
				],
				
				'selectors' => [
					'{{WRAPPER}} .post-block-item-three .news-post-grid-thumbnail a img' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);		
		
		
		$this->add_control(
            'enable_reverse_column',
            [
                'label' => esc_html__('Enable Image & Content Reverse', 'vibenews-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'vibenews-extra'),
                'label_off' => esc_html__('No', 'vibenews-extra'),
                'default' => 'no',
            ]
        );
		
		
		
		$this->end_controls_section();
	
	}
	
	/**
    * Post Query Options
    */
    private function post_query_options() {
	
	
		$this->start_controls_section(
            'post_query_option',
            [
                'label' => __( 'Post Options', 'vibenews-extra' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
		
		
		// Post Sort
		
        $this->add_control(
            'post_sorting',
            [
                'type'    => \Elementor\Controls_Manager::SELECT2,
                'label' => esc_html__('Post Sorting', 'vibenews-extra'),
                'default' => 'date',
                'options' => [
					'date' => esc_html__('Recent Post', 'vibenews-extra'),
                    'rand' => esc_html__('Random Post', 'vibenews-extra'),
					'title'         => __( 'Title Sorting Post', 'vibenews-extra' ),
                    'modified' => esc_html__('Last Modified Post', 'vibenews-extra'),
                    'comment_count' => esc_html__('Most Commented Post', 'vibenews-extra'),
					
                ],
            ]
        );		
		
		// Post Order
		
        $this->add_control(
            'post_ordering',
            [
                'type'    => \Elementor\Controls_Manager::SELECT2,
                'label' => esc_html__('Post Ordering', 'vibenews-extra'),
                'default' => 'DESC',
                'options' => [
					'DESC' => esc_html__('Desecending', 'vibenews-extra'),
                    'ASC' => esc_html__('Ascending', 'vibenews-extra'),
                ],
            ]
        );
		
		
		// Post Categories
		
		$this->add_control(
            'post_categories',
            [
                'type'      => \Elementor\Controls_Manager::SELECT2,
				'label' =>esc_html__('Select Categories', 'vibenews-extra'),
                'options'   => $this->posts_cat_list(),
                'label_block' => true,
                'multiple'  => true,
            ]
        );
	
		$this->add_control(
			'post_tags',
			[
				'label'       => esc_html__('Select Tags', 'vibenews-extra'),
				'type'      => \Elementor\Controls_Manager::SELECT2,
				'options'     => vibenews_post_tags(),
				'label_block' => true,
				'multiple'    => true,
			]
		);
		
		
		// Post Items.
		
        $this->add_control(
            'post_number',
			[
				'label'         => esc_html__( 'Number Of Posts', 'vibenews-extra' ),
				'type'          => \Elementor\Controls_Manager::NUMBER,
				'default'       => '2',
			]
        );
		
		$this->add_control(
            'enable_offset_post',
            [
               'label' => esc_html__('Enable Skip Post', 'vibenews-extra'),
               'type' => \Elementor\Controls_Manager::SWITCHER,
               'label_on' => esc_html__('Yes', 'vibenews-extra'),
               'label_off' => esc_html__('No', 'vibenews-extra'),
               'default' => 'no',
               
            ]
        );
      
        $this->add_control(
			'post_offset_count',
			  [
			   'label'         => esc_html__( 'Skip Post Count', 'vibenews-extra' ),
			   'type'          => \Elementor\Controls_Manager::NUMBER,
			   'default'       => '1',
			   'condition' => [ 'enable_offset_post' => 'yes' ]

			  ]
        );
		
		
		$this->end_controls_section();
	
	}	
	
	/**
    * Meta Options
    */
    private function meta_options() {
	
	
		$this->start_controls_section(
            'meta_option',
            [
                'label' => __( 'Meta Options', 'vibenews-extra' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
		
		$this->add_control(
            'display_excerpt',
            [
                'label' => esc_html__('Display Post Excerpt', 'ennlil-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'ennlil-extra'),
                'label_off' => esc_html__('No', 'ennlil-extra'),
                'default' => 'yes',
            ]
        );

		$this->add_control(
         	'display_author',
         	[
				 'label' => esc_html__('Display Author', 'vibenews-extra'),
				 'type' => \Elementor\Controls_Manager::SWITCHER,
				 'label_on' => esc_html__('Yes', 'vibenews-extra'),
				 'label_off' => esc_html__('No', 'vibenews-extra'),
				 'default' => 'yes',
         	]
     	);

     	$this->add_control(
            'display_cat',
            [
                'label' => esc_html__('Display Category Name', 'vibenews-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'vibenews-extra'),
                'label_off' => esc_html__('No', 'vibenews-extra'),
                'default' => 'yes',
            ]
        );
		
		$this->add_control(
            'display_date',
            [
                'label' => esc_html__('Display Date', 'vibenews-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'vibenews-extra'),
                'label_off' => esc_html__('No', 'vibenews-extra'),
                'default' => 'yes',
            ]
        );
		
		
		$this->add_control(
            'display_read_time',
            [
                'label' => esc_html__('Display Post Read Time', 'vibenews-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'vibenews-extra'),
                'label_off' => esc_html__('No', 'vibenews-extra'),
                'default' => 'no',
            ]
        );

 
		
	
		$this->end_controls_section();
	
	}	
	
	/**
    * Design Options
    */
    private function design_options() {
	
	
		$this->start_controls_section(
            'design_option',
            [
                'label' => __( 'Block Typography', 'vibenews-extra' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
		

		$this->add_group_control(
           \Elementor\Group_Control_Typography::get_type(),
           [
              'name' => 'block_top_title_typography',
              'label' => esc_html__( 'Top Title Typography', 'vibenews-extra' ),
              'selector' => '{{WRAPPER}} .theme-post-block-three-wrapper .post-block-title-wrapper h2.post-block-title',
           ]
        );	
		
		
		$this->add_group_control(
           \Elementor\Group_Control_Typography::get_type(),
           [
              'name' => 'block_three_title_typography',
              'label' => esc_html__( 'Post Title Typography', 'vibenews-extra' ),
              'selector' => '{{WRAPPER}} .post-block-item-three .grid-content-top h3.post-title',
           ]
        );		
		
		
		$this->add_group_control(
           \Elementor\Group_Control_Typography::get_type(),
           [
              'name' => 'block_three_excerpt_typography',
              'label' => esc_html__( 'Post Excerpt Typography', 'vibenews-extra' ),
              'selector' => '{{WRAPPER}} .post-block-item-three .grid-content-top .post-excerpt-box p',
           ]
        );	
		
		$this->add_group_control(
           \Elementor\Group_Control_Typography::get_type(),
           [
              'name' => 'three_cat_meta_typography',
              'label' => esc_html__( 'Category Meta Typography', 'vibenews-extra' ),
              'selector' => '{{WRAPPER}} .post-block-item-three .post-cat-box.post-block-meta-cat a.news-cat_Name',
           ]
        );

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
			   'name' => 'three_date_box_typography',
			   'label' => esc_html__( 'Date Meta Typography', 'vibenews-extra' ),
			   'selector' => '{{WRAPPER}} .post-block-item-three .post-date-box.post-block-meta-date',
			]
		 );
		
		$this->add_control('block_top_title_color', [
            'label' => esc_html__('Top Title Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .theme-post-block-three-wrapper .post-block-title-wrapper h2.post-block-title" => "color: {{VALUE}}"
            ]
        ]);
		
		$this->add_control('block_three_title_color', [
            'label' => esc_html__('Post Title Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .post-block-item-three .grid-content-top h3.post-title a" => "color: {{VALUE}}"
            ]
        ]);
		
		$this->add_control('block_three_excerpt_color', [
            'label' => esc_html__('Post Excerpt Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .post-block-item-three .grid-content-top .post-excerpt-box p" => "color: {{VALUE}}"
            ]
        ]);
		
		$this->add_control('block_three_cat_color', [
            'label' => esc_html__('Post Category Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .post-block-item-three .post-cat-box.post-block-meta-cat a.news-cat_Name" => "color: {{VALUE}}"
            ]
        ]);
		
		$this->add_control('block_three_date_color', [
            'label' => esc_html__('Post Date Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .post-block-item-three .post-date-box.post-block-meta-date" => "color: {{VALUE}}"
            ]
        ]);
		

		$this->add_control(
          'title_length',
          [
            'label'         => esc_html__( 'Post Title Length', 'vibenews-extra' ),
            'type'          => \Elementor\Controls_Manager::NUMBER,
            'default'       => '10',
          ]
        );

        $this->add_control(
          'content_length',
          [
            'label'         => esc_html__( 'Post Excerpt Length', 'ennlil-extra' ),
            'type'          => \Elementor\Controls_Manager::NUMBER,
            'default'       => '20',
          ]
        );
		

		
		$this->end_controls_section();
	
	}	
		


	protected function render() {
		
		
		$settings = $this->get_settings_for_display();

		$title_limit = $settings['title_length'];
		$content_limit = $settings['content_length'];
		$post_count = $settings['post_number'];
		$post_order  = $settings['post_ordering'];
		$post_sortby = $settings['post_sorting']; 
		
		$display_blog_excerpt = $settings['display_excerpt'];
		$display_blog_cat = $settings['display_cat'];
		$display_blog_author = $settings['display_author'];
		$display_blog_date = $settings['display_date'];
		
		$display_top_title = $settings['display_three_top_title'];
		
		$enable_reverse_column = $settings['enable_reverse_column'];

		
		$display_read_time = $settings['display_read_time'];

		
		$args = [
            'post_type' => 'post',
            'post_status' => 'publish',
			'order' => $settings['post_ordering'],
			'posts_per_page' => $settings['post_number'],
			'tag__in'        => $settings['post_tags'],
			'ignore_sticky_posts' => 1,
			'suppress_filters' => false,
        ];
		
		// Category
        if ( ! empty( $settings['post_categories'] ) ) {
            $args['category_name'] = implode(',', $settings['post_categories']);
        }
		
		// Post Sorting
        if ( ! empty( $settings['post_sorting'] ) ) {
            $args['orderby'] = $settings['post_sorting'];
        }	
		
		// Post Offset		
		if($settings['enable_offset_post'] == 'yes') {
			$args['offset'] = $settings['post_offset_count'];
		}



		// Query
        $query = new \WP_Query( $args ); ?>
		
		
	<div class="theme-post-block-three-wrapper">
	
		<?php if($display_top_title=='yes'): ?>	
		<div class="post-block-title-wrapper">
			<h2 class="post-block-title"><?php echo wp_kses_post( $settings['post_block_title']  ); ?></h2>
		</div>
		<?php endif; ?>
		
		<?php if ( $query->have_posts() ) : ?>

			<?php while ($query->have_posts()) : $query->the_post(); ?>
				
				<article <?php post_class( 'post-block-three-wrap' ); ?>>
				
				<div class="post-block-item-three <?php if( $enable_reverse_column == "yes" ) { echo "theme-col-reverse"; } else { echo "theme-col-normal"; } ?>">
					
					<div class="news-post-grid-thumbnail">
						<a href="<?php the_permalink(); ?>" class="news-post-grid-thumbnail-wrap">
							<img src="<?php echo esc_attr(esc_url(get_the_post_thumbnail_url(null, 'full'))); ?>" alt="<?php the_title_attribute(); ?>">
						</a>
					</div>
					
					<div class="grid-content-top post-block-item-three-inner">
					
						<div class="blog-post-meta-items post-block-meta-top">
							<?php if($display_blog_cat=='yes'): ?>	
							<div class="post-cat-box post-block-meta-cat">
							<?php require VIBENEWS_THEME_DIR . '/template-parts/cat-color.php'; ?>
							</div>
							<?php endif; ?>
							
							<?php if($display_blog_date=='yes'): ?>
							<div class="post-date-box post-block-meta-date">
							<?php echo esc_html( get_the_date( 'F j, Y' ) ); ?>
							</div>
							<?php endif; ?>	
						</div>
						
						<h3 class="post-title">
							<a href="<?php the_permalink(); ?>"><?php echo esc_html( wp_trim_words(get_the_title(), $title_limit,'') ); ?></a>
						</h3>
					
						<?php if($display_blog_excerpt =='yes'): ?> 
						<div class="post-excerpt-box">
							<p><?php echo esc_html( wp_trim_words(get_the_excerpt(), $content_limit ,'') );?></p>
						</div>
						<?php endif; ?>	
						
					</div>
					
				</div>
				
				</article>

			<?php endwhile; ?>
		
		<?php wp_reset_postdata(); ?>
		
		<?php endif; ?>
		
	
	</div>
	
	<?php 
}
		
   
   	public function posts_cat_list() {
		
		$terms = get_terms( array(
			'taxonomy'    => 'category',
			'hide_empty'  => true
		) );

		$cat_list = [];
		foreach($terms as $post) {
		$cat_list[$post->slug]  = [$post->name];
		}
		return $cat_list;
	  
	}		
	
}

