<?php

/**
 * Elementor Widget
 * @package Vibenews
 * @since 1.0.0
 */


if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly.

class vibenews_post_block_video extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve Elementor widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'posts-video-block';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Elementor widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Theme Video Blocks', 'vibenews-extra' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Elementor widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'fas fa-th-large'; 
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Elementor widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'vibenews_widgets' ];
	}

	/**
	 * Register Elementor widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	 
	
	protected function register_controls() {
		$this->layout_options();		
		$this->post_query_options();	
		$this->meta_options();	
		$this->design_options();
	}
	
	/**
    * Layout Options
    */
    private function layout_options() {
	
	
		$this->start_controls_section(
            'layout_option',
            [
                'label' => __( 'Layout Options', 'vibenews-extra' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
		
		
		$this->add_responsive_control(
			'video_block_img_height',
			[
				'label' =>esc_html__( 'Set Post Image Height', 'vibenews-extra' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
					],
				],
				'devices' => [ 'desktop', 'tablet', 'mobile' ],
				'desktop_default' => [
					'size' => 360,
					'unit' => 'px',
				],
				'tablet_default' => [
					'size' => 270,
					'unit' => 'px',
				],
				'mobile_default' => [
					'size' => 250,
					'unit' => 'px',
				],
				
				'default'  => [
					'unit' => 'px',
					'size' => 360,
				],
				
				'selectors' => [
					'{{WRAPPER}} .theme-post-video-wrapper.theme-post-video-block .video-block-post-image img' => 'height: {{SIZE}}{{UNIT}}!important;',
				],
			]
		);		
		
		$this->add_control(
            'display_top_title',
            [
                'label' => esc_html__('Display Top Heading', 'vibenews-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'vibenews-extra'),
                'label_off' => esc_html__('No', 'vibenews-extra'),
                'default' => 'yes',
            ]
        );
		
		$this->add_control(
			'video_post_block_title',
			[
				'label' => __( 'Section Title', 'vibenews-extra' ),
				'type' => \Elementor\Controls_Manager::TEXTAREA,
				'default' => __( 'Videos', 'vibenews-extra' ),
				'placeholder' => __( 'Type Your title here', 'vibenews-extra' ),
			]
		);
		
		
		
		$this->end_controls_section();
	
	}
	
	/**
    * Post Query Options
    */
    private function post_query_options() {
	
	
		$this->start_controls_section(
            'post_query_option',
            [
                'label' => __( 'Post Options', 'vibenews-extra' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
		
		
		// Post Sort
		
        $this->add_control(
            'post_sorting',
            [
                'type'    => \Elementor\Controls_Manager::SELECT2,
                'label' => esc_html__('Post Sorting', 'vibenews-extra'),
                'default' => 'date',
                'options' => [
					'date' => esc_html__('Recent Post', 'vibenews-extra'),
                    'rand' => esc_html__('Random Post', 'vibenews-extra'),
					'title'         => __( 'Title Sorting Post', 'vibenews-extra' ),
                    'modified' => esc_html__('Last Modified Post', 'vibenews-extra'),
                    'comment_count' => esc_html__('Most Commented Post', 'vibenews-extra'),
					
                ],
            ]
        );		
		
		// Post Order
		
        $this->add_control(
            'post_ordering',
            [
                'type'    => \Elementor\Controls_Manager::SELECT2,
                'label' => esc_html__('Post Ordering', 'vibenews-extra'),
                'default' => 'DESC',
                'options' => [
					'DESC' => esc_html__('Desecending', 'vibenews-extra'),
                    'ASC' => esc_html__('Ascending', 'vibenews-extra'),
                ],
            ]
        );
		
		
		// Post Categories
		
		$this->add_control(
            'post_categories',
            [
                'type'      => \Elementor\Controls_Manager::SELECT2,
				'label' =>esc_html__('Select Categories', 'vibenews-extra'),
                'options'   => $this->posts_cat_list(),
                'label_block' => true,
                'multiple'  => true,
            ]
        );
	
		
		
		// Post Items.
		
        $this->add_control(
            'post_number',
			[
				'label'         => esc_html__( 'Number Of Posts', 'vibenews-extra' ),
				'type'          => \Elementor\Controls_Manager::NUMBER,
				'default'       => '1',
			]
        );
		
		$this->add_control(
            'enable_offset_post',
            [
               'label' => esc_html__('Enable Skip Post', 'vibenews-extra'),
               'type' => \Elementor\Controls_Manager::SWITCHER,
               'label_on' => esc_html__('Yes', 'vibenews-extra'),
               'label_off' => esc_html__('No', 'vibenews-extra'),
               'default' => 'no',
               
            ]
        );
      
        $this->add_control(
			'post_offset_count',
			  [
			   'label'         => esc_html__( 'Skip Post Count', 'vibenews-extra' ),
			   'type'          => \Elementor\Controls_Manager::NUMBER,
			   'default'       => '1',
			   'condition' => [ 'enable_offset_post' => 'yes' ]

			  ]
        );
		
		
		$this->end_controls_section();
	
	}	
	
	/**
    * Meta Options
    */
    private function meta_options() {
	
	
		$this->start_controls_section(
            'meta_option',
            [
                'label' => __( 'Meta Options', 'vibenews-extra' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
		
		
		$this->add_control(
            'display_excerpt',
            [
                'label' => esc_html__('Display Post Excerpt', 'ennlil-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'ennlil-extra'),
                'label_off' => esc_html__('No', 'ennlil-extra'),
                'default' => 'yes',
            ]
        );

		$this->add_control(
         	'display_author',
         	[
				 'label' => esc_html__('Display Author', 'vibenews-extra'),
				 'type' => \Elementor\Controls_Manager::SWITCHER,
				 'label_on' => esc_html__('Yes', 'vibenews-extra'),
				 'label_off' => esc_html__('No', 'vibenews-extra'),
				 'default' => 'yes',
         	]
     	);

     	$this->add_control(
            'display_cat',
            [
                'label' => esc_html__('Display Category Name', 'vibenews-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'vibenews-extra'),
                'label_off' => esc_html__('No', 'vibenews-extra'),
                'default' => 'yes',
            ]
        );
		
		$this->add_control(
            'display_date',
            [
                'label' => esc_html__('Display Date', 'vibenews-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'vibenews-extra'),
                'label_off' => esc_html__('No', 'vibenews-extra'),
                'default' => 'yes',
            ]
        );
		

		
		$this->add_control(
            'display_read_time',
            [
                'label' => esc_html__('Display Post Read Time', 'vibenews-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'vibenews-extra'),
                'label_off' => esc_html__('No', 'vibenews-extra'),
                'default' => 'no',
            ]
        );

 
		
	
		$this->end_controls_section();
	
	}	
	
	/**
    * Design Options
    */
    private function design_options() {
	
	
		$this->start_controls_section(
            'design_option',
            [
                'label' => __( 'Typography', 'vibenews-extra' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
		
		
		$this->add_group_control(
           \Elementor\Group_Control_Typography::get_type(),
           [
              'name' => 'block_video_sec_title_typography',
              'label' => esc_html__( 'Top Title Typography', 'vibenews-extra' ),
              'selector' => '{{WRAPPER}} .theme-post-video-wrapper.theme-post-video-block .post-block-title-wrapper h2.post-block-title',
           ]
        );	
		
		
		$this->add_group_control(
           \Elementor\Group_Control_Typography::get_type(),
           [
              'name' => 'video_post_title_typography',
              'label' => esc_html__( 'Post Title Typography', 'vibenews-extra' ),
              'selector' => '{{WRAPPER}} .theme-post-video-wrapper.theme-post-video-block .video-post-block-content h3.post-title',
           ]
        );		
		
		
		$this->add_group_control(
           \Elementor\Group_Control_Typography::get_type(),
           [
              'name' => 'video_post_excerpt_typography',
              'label' => esc_html__( 'Post Excerpt Typography', 'vibenews-extra' ),
              'selector' => '{{WRAPPER}} .theme-post-video-wrapper.theme-post-video-block .video-post-block-content .post-excerpt-box p',
           ]
        );	
		
		$this->add_control('video_post_top_title_color', [
            'label' => esc_html__('Top Title Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .theme-post-video-wrapper.theme-post-video-block .post-block-title-wrapper h2.post-block-title" => "color: {{VALUE}}"
            ]
        ]);
		
		$this->add_control('video_post_title_color', [
            'label' => esc_html__('Post Title Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .theme-post-video-wrapper.theme-post-video-block .video-post-block-content h3.post-title a" => "color: {{VALUE}}"
            ]
        ]);
		
		$this->add_control('video_post_excerpt_color', [
            'label' => esc_html__('Post Excerpt Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .theme-post-video-wrapper.theme-post-video-block .video-post-block-content .post-excerpt-box p     " => "color: {{VALUE}}"
            ]
        ]);
		

		
		$this->add_control(
          'title_length',
          [
            'label'         => esc_html__( 'Post Title Length', 'vibenews-extra' ),
            'type'          => \Elementor\Controls_Manager::NUMBER,
            'default'       => '10',
          ]
        );
		
		$this->add_control(
          'content_length',
          [
            'label'         => esc_html__( 'Post Excerpt Length', 'ennlil-extra' ),
            'type'          => \Elementor\Controls_Manager::NUMBER,
            'default'       => '20',
          ]
        );
		
		
		
		$this->end_controls_section();
	
	}	
		


	protected function render() {
		
		
		$settings = $this->get_settings_for_display();
	
		
		$title_limit = $settings['title_length'];
		$content_limit = $settings['content_length'];
		
		$post_count = $settings['post_number'];
		$post_order  = $settings['post_ordering'];
		$post_sortby = $settings['post_sorting']; 
		
		$display_blog_cat = $settings['display_cat'];
		
		$display_blog_excerpt = $settings['display_excerpt'];
		
		$display_blog_author = $settings['display_author'];
		$display_blog_date = $settings['display_date'];
		$display_read_time = $settings['display_read_time'];
		
		$display_top_title = $settings['display_top_title'];

		
		$args = [
            'post_type' => 'post',
            'post_status' => 'publish',
			'ignore_sticky_posts' => 1,
			'tax_query' => [[
                'taxonomy' => 'post_format',
                'field' => 'slug',
                'terms' => array( 'post-format-video'),
                'operator' => 'IN'
            ]]
        ];
		
		
		// How many posts
        if ( ! empty( $settings['post_number'] ) ) {
            $args['posts_per_page'] = $settings['post_number'];
        }
		
		
		// Category
        if ( ! empty( $settings['post_categories'] ) ) {
            $args['category_name'] = implode(',', $settings['post_categories']);
        }
		
		// Post Sorting
        if ( ! empty( $settings['post_sorting'] ) ) {
            $args['orderby'] = $settings['post_sorting'];
        }	
		
		// Post Ordering
        if ( ! empty( $settings['post_ordering'] ) ) {
            $args['order'] = $settings['post_ordering'];
        }	
		
		
		// Post Offset		
		if($settings['enable_offset_post'] == 'yes') {
			$args['offset'] = $settings['post_offset_count'];
		}

		// Query
        $query = new \WP_Query( $args ); ?>
		
		
	<div class="theme-post-video-wrapper theme-post-video-block">
			
		<?php if($display_top_title=='yes'): ?>	
		<div class="post-block-title-wrapper">
			<h2 class="post-block-title"><span><?php echo wp_kses_post( $settings['video_post_block_title']  ); ?></span></h2>
		</div>
		<?php endif; ?>
		
		
		<?php if ( $query->have_posts() ) : ?>
		
        <?php while ($query->have_posts()) : $query->the_post(); ?>
			
			<div class="theme-video_blocks video-box-wrapper">
					
				<div class="blog-post-video-thumbnail">
					<div class="video-block-post-image">
						<img src="<?php echo esc_url(get_the_post_thumbnail_url()); ?>" alt="<?php the_title_attribute(); ?>">
					</div>
					
					<div class="video-play-icon-wrap">
						<a href="<?php 
						$theme_video_meta = get_post_meta(get_the_ID(),'theme_postvideo_options',true);
						$video_url = isset($theme_video_meta['textm']) && !empty($theme_video_meta['textm']) ? $theme_video_meta['textm'] : '';
						echo esc_url($video_url);?>" class="theme-play-btn video-play-btn">
						<i class="icofont-ui-play" aria-hidden="true"></i>
						</a>
					</div>
				</div>
				
				
				<div class="video-post-block-content">
					
					<h3 class="post-title">
						<a href="<?php the_permalink(); ?>"><?php echo esc_html( wp_trim_words(get_the_title(), $title_limit,'') ); ?></a>
					</h3>
				
					<?php if($display_blog_excerpt =='yes'): ?> 
					<div class="post-excerpt-box">
						<p><?php echo esc_html( wp_trim_words(get_the_excerpt(), $content_limit ,'') );?></p>
					</div>
					<?php endif; ?>	
					

				</div>

			</div>
			

		<?php endwhile; ?>
		
		
		<?php wp_reset_postdata(); ?>
		
		<?php endif; ?>
		
		</div>
		
	
		<?php 
      }
		
   
   	public function posts_cat_list() {
		
		$terms = get_terms( array(
			'taxonomy'    => 'category',
			'hide_empty'  => true
		) );

		$cat_list = [];
		foreach($terms as $post) {
		$cat_list[$post->slug]  = [$post->name];
		}
		return $cat_list;
	  
	}		
	
}

