<?php

/**
 * Elementor Widget
 * @package Vibenews
 * @since 1.0.0
 */


if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly.

class vibenews_post_block_list extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve Elementor widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'posts-blocklist';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Elementor widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Posts Block List', 'vibenews-extra' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Elementor widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'fas fa-th-large'; 
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Elementor widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'vibenews_widgets' ];
	}

	/**
	 * Register Elementor widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	 
	
	protected function register_controls() {
		$this->layout_options();		
		$this->post_query_options();	
		$this->meta_options();	
		$this->design_options();
	}
	
	/**
    * Layout Options
    */
    private function layout_options() {
	
	
		$this->start_controls_section(
            'layout_option',
            [
                'label' => __( 'Layout Options', 'vibenews-extra' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
		
		
		$this->add_responsive_control(
			'grid_img_height',
			[
				'label' =>esc_html__( 'Set Top Post Image Height', 'vibenews-extra' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
					],
				],
				'devices' => [ 'desktop', 'tablet', 'mobile' ],
				'desktop_default' => [
					'size' => 212,
					'unit' => 'px',
				],
				'tablet_default' => [
					'size' => 212,
					'unit' => 'px',
				],
				'mobile_default' => [
					'size' => 212,
					'unit' => 'px',
				],
				
				'default'  => [
					'unit' => 'px',
					'size' => 212,
				],
				
				'selectors' => [
					'{{WRAPPER}} .cutsom-post-block-list-wrapper .post-block-list-wrap .news-post-grid-thumbnail a img' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);		
		
		$this->add_control(
			'post_list_sec_title',
			[
				'label' => __( 'Section Title', 'vibenews-extra' ),
				'type' => \Elementor\Controls_Manager::TEXTAREA,
				'default' => __( 'Just In', 'vibenews-extra' ),
				'placeholder' => __( 'Type Your title here', 'vibenews-extra' ),
			]
		);
		
		$this->add_control(
            'post_list_title_enable',
            [
                'label' => esc_html__('Display Top Heading', 'vibenews-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'vibenews-extra'),
                'label_off' => esc_html__('No', 'vibenews-extra'),
                'default' => 'yes',
            ]
        );
		
		
		
		$this->end_controls_section();
	
	}
	
	/**
    * Post Query Options
    */
    private function post_query_options() {
	
	
		$this->start_controls_section(
            'post_query_option',
            [
                'label' => __( 'Post Options', 'vibenews-extra' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
		
		
		// Post Sort
		
        $this->add_control(
            'post_sorting',
            [
                'type'    => \Elementor\Controls_Manager::SELECT2,
                'label' => esc_html__('Post Sorting', 'vibenews-extra'),
                'default' => 'date',
                'options' => [
					'date' => esc_html__('Recent Post', 'vibenews-extra'),
                    'rand' => esc_html__('Random Post', 'vibenews-extra'),
					'title'         => __( 'Title Sorting Post', 'vibenews-extra' ),
                    'modified' => esc_html__('Last Modified Post', 'vibenews-extra'),
                    'comment_count' => esc_html__('Most Commented Post', 'vibenews-extra'),
					
                ],
            ]
        );		
		
		// Post Order
		
        $this->add_control(
            'post_ordering',
            [
                'type'    => \Elementor\Controls_Manager::SELECT2,
                'label' => esc_html__('Post Ordering', 'vibenews-extra'),
                'default' => 'DESC',
                'options' => [
					'DESC' => esc_html__('Desecending', 'vibenews-extra'),
                    'ASC' => esc_html__('Ascending', 'vibenews-extra'),
                ],
            ]
        );
		
		
		// Post Categories
		
		$this->add_control(
            'post_categories',
            [
                'type'      => \Elementor\Controls_Manager::SELECT2,
				'label' =>esc_html__('Select Categories', 'vibenews-extra'),
                'options'   => $this->posts_cat_list(),
                'label_block' => true,
                'multiple'  => true,
            ]
        );
	
		$this->add_control(
			'post_tags',
			[
				'label'       => esc_html__('Select Tags', 'vibenews-extra'),
				'type'      => \Elementor\Controls_Manager::SELECT2,
				'options'     => vibenews_post_tags(),
				'label_block' => true,
				'multiple'    => true,
			]
		);
		
		
		// Post Items.
		
        $this->add_control(
            'post_number',
			[
				'label'         => esc_html__( 'Number Of Posts', 'vibenews-extra' ),
				'type'          => \Elementor\Controls_Manager::NUMBER,
				'default'       => '1',
			]
        );
		

		// Specific Posts by ID.
        $this->add_control(
            'post_ids',
            [
                'type'        => \Elementor\Controls_Manager::TEXT,
                'label'       => __( 'Show specific posts by ID', 'deo-elementor' ),
                'placeholder' => __( 'ex.: 256, 54, 78', 'deo-elementor' ),
                'description'   => __( 'Paste post ID\'s separated by commas. To find ID, click edit post and you\'ll find it in the browser address bar', 'deo-elementor' ),
                'default'     => '',
                'separator'     => 'before',
                'label_block' => true,
            ]
        );


		$this->add_control(
            'enable_offset_post',
            [
               'label' => esc_html__('Enable Skip Post', 'vibenews-extra'),
               'type' => \Elementor\Controls_Manager::SWITCHER,
               'label_on' => esc_html__('Yes', 'vibenews-extra'),
               'label_off' => esc_html__('No', 'vibenews-extra'),
               'default' => 'no',
               
            ]
        );
      
        $this->add_control(
			'post_offset_count',
			  [
			   'label'         => esc_html__( 'Skip Post Count', 'vibenews-extra' ),
			   'type'          => \Elementor\Controls_Manager::NUMBER,
			   'default'       => '1',
			   'condition' => [ 'enable_offset_post' => 'yes' ]

			  ]
        );
		
		
		$this->end_controls_section();
	
	}	
	
	/**
    * Meta Options
    */
    private function meta_options() {
	
	
		$this->start_controls_section(
            'meta_option',
            [
                'label' => __( 'Meta Options', 'vibenews-extra' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
		
		$this->add_control(
            'display_excerpt',
            [
                'label' => esc_html__('Display Post Excerpt', 'ennlil-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'ennlil-extra'),
                'label_off' => esc_html__('No', 'ennlil-extra'),
                'default' => 'yes',
            ]
        );
		
		$this->add_control(
            'display_excerpt_small',
            [
                'label' => esc_html__('Display Small Post Excerpt', 'ennlil-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'ennlil-extra'),
                'label_off' => esc_html__('No', 'ennlil-extra'),
                'default' => 'yes',
            ]
        );

		$this->add_control(
         	'display_author',
         	[
				 'label' => esc_html__('Display Author', 'vibenews-extra'),
				 'type' => \Elementor\Controls_Manager::SWITCHER,
				 'label_on' => esc_html__('Yes', 'vibenews-extra'),
				 'label_off' => esc_html__('No', 'vibenews-extra'),
				 'default' => 'yes',
         	]
     	);

     	$this->add_control(
            'display_cat',
            [
                'label' => esc_html__('Display Category Name', 'vibenews-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'vibenews-extra'),
                'label_off' => esc_html__('No', 'vibenews-extra'),
                'default' => 'yes',
            ]
        );
		
		$this->add_control(
            'display_cat_small',
            [
                'label' => esc_html__('Display Small List Category Name', 'vibenews-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'vibenews-extra'),
                'label_off' => esc_html__('No', 'vibenews-extra'),
                'default' => 'yes',
            ]
        );
		
		$this->add_control(
            'display_date',
            [
                'label' => esc_html__('Display Date', 'vibenews-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'vibenews-extra'),
                'label_off' => esc_html__('No', 'vibenews-extra'),
                'default' => 'yes',
            ]
        );
		
		$this->add_control(
            'display_date_small',
            [
                'label' => esc_html__('Display Small List Date', 'vibenews-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'vibenews-extra'),
                'label_off' => esc_html__('No', 'vibenews-extra'),
                'default' => 'yes',
            ]
        );
		
		$this->add_control(
            'blog_comment',
            [
                'label' => esc_html__('Display Post Comment Number', 'vibenews-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'vibenews-extra'),
                'label_off' => esc_html__('No', 'vibenews-extra'),
                'default' => 'yes',
            ]
        );
		
		$this->add_control(
            'display_read_time',
            [
                'label' => esc_html__('Display Post Read Time', 'vibenews-extra'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'vibenews-extra'),
                'label_off' => esc_html__('No', 'vibenews-extra'),
                'default' => 'no',
            ]
        );

 
		
	
		$this->end_controls_section();
	
	}	
	
	/**
    * Design Options
    */
    private function design_options() {
	
	
		$this->start_controls_section(
            'design_option',
            [
                'label' => __( 'Slider Typography', 'vibenews-extra' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
		
		

		$this->add_group_control(
           \Elementor\Group_Control_Typography::get_type(),
           [
              'name' => 'postlist_top_title_typography',
              'label' => esc_html__( 'Top Title Typography', 'vibenews-extra' ),
              'selector' => '{{WRAPPER}} .cutsom-post-block-list-wrapper .post-block-title-wrapper h2.post-block-title',
           ]
        );	
		
		
		$this->add_group_control(
           \Elementor\Group_Control_Typography::get_type(),
           [
              'name' => 'postlist_title_typography',
              'label' => esc_html__( 'Post Title Typography', 'vibenews-extra' ),
              'selector' => '{{WRAPPER}} .cutsom-post-block-list-inner .news-post-grid-content.grid-content-inner h3.post-title',
           ]
        );	
		
		$this->add_group_control(
           \Elementor\Group_Control_Typography::get_type(),
           [
              'name' => 'postlist_small_title_typography',
              'label' => esc_html__( 'Small Post Title Typography', 'vibenews-extra' ),
              'selector' => '{{WRAPPER}} .cutsom-post-block-list-inner .grid-content-inner.post-block-list-small-inner h3.post-title',
           ]
        );		
		
		
		$this->add_group_control(
           \Elementor\Group_Control_Typography::get_type(),
           [
              'name' => 'postlist_excerpt_typography',
              'label' => esc_html__( 'Post Excerpt Typography', 'vibenews-extra' ),
              'selector' => '{{WRAPPER}} .cutsom-post-block-list-inner .news-post-grid-content.grid-content-inner .post-excerpt-box p',
           ]
        );	
		
		$this->add_group_control(
           \Elementor\Group_Control_Typography::get_type(),
           [
              'name' => 'postlist_small_excerpt_typography',
              'label' => esc_html__( 'Post Excerpt Typography', 'vibenews-extra' ),
              'selector' => '{{WRAPPER}} .cutsom-post-block-list-inner .grid-content-inner.post-block-list-small-inner .post-excerpt-box p',
           ]
        );	
		
		$this->add_group_control(
           \Elementor\Group_Control_Typography::get_type(),
           [
              'name' => 'postlist_topcat_meta_typography',
              'label' => esc_html__( 'Category Meta Typography', 'vibenews-extra' ),
              'selector' => '{{WRAPPER}} .cutsom-post-block-list-inner .post-cat-box.post-block-meta-cat a.news-cat_Name',
           ]
        );


		$this->add_group_control(
           \Elementor\Group_Control_Typography::get_type(),
           [
              'name' => 'postlist_smallcat_meta_typography',
              'label' => esc_html__( 'Small Category Meta Typography', 'vibenews-extra' ),
              'selector' => '{{WRAPPER}} .cutsom-post-block-list-inner .post-block-list-small-inner .post-cat-box.post-block-meta-cat a.news-cat_Name',
           ]
        );
		
		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
			   'name' => 'poslist_large_date_box_typography',
			   'label' => esc_html__( 'Date Meta Typography', 'vibenews-extra' ),
			   'selector' => '{{WRAPPER}} .cutsom-post-block-list-inner .post-date-box.post-block-meta-date',
			]
		 );
		 
		 $this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
			   'name' => 'poslist_small_date_box_typography',
			   'label' => esc_html__( 'Small Date Meta Typography', 'vibenews-extra' ),
			   'selector' => '{{WRAPPER}} .cutsom-post-block-list-inner .post-block-list-small-inner .post-date-box.post-block-meta-date',
			]
		 );
		 
		$this->add_control('postlist_top_title_color', [
            'label' => esc_html__('Top Title Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .cutsom-post-block-list-wrapper .post-block-title-wrapper h2.post-block-title" => "color: {{VALUE}}"
            ]
        ]);
		
		$this->add_control('postlist_large_title_color', [
            'label' => esc_html__('Large Post Title Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .cutsom-post-block-list-inner .news-post-grid-content.grid-content-inner h3.post-title a" => "color: {{VALUE}}"
            ]
        ]);
		
		$this->add_control('postlist_small_title_color', [
            'label' => esc_html__('Small Post Title Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .cutsom-post-block-list-inner .grid-content-inner.post-block-list-small-inner h3.post-title a" => "color: {{VALUE}}"
            ]
        ]);
		
		$this->add_control('postlist_large_excerpt_color', [
            'label' => esc_html__('Post Excerpt Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .cutsom-post-block-list-inner .news-post-grid-content.grid-content-inner .post-excerpt-box p" => "color: {{VALUE}}"
            ]
        ]);
		
		$this->add_control('postlist_small_excerpt_color', [
            'label' => esc_html__('Small Post Excerpt Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .cutsom-post-block-list-inner .grid-content-inner.post-block-list-small-inner .post-excerpt-box p" => "color: {{VALUE}}"
            ]
        ]);
		
		$this->add_control('postlist_large_cat_color', [
            'label' => esc_html__('Post Category Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .cutsom-post-block-list-inner .post-cat-box.post-block-meta-cat a.news-cat_Name" => "color: {{VALUE}}"
            ]
        ]);	
		
		$this->add_control('postlist_small_cat_color', [
            'label' => esc_html__('Small Post Category Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .cutsom-post-block-list-inner .post-block-list-small-inner .post-cat-box.post-block-meta-cat a.news-cat_Name" => "color: {{VALUE}}"
            ]
        ]);
		
		$this->add_control('postlist_large_date_color', [
            'label' => esc_html__('Post Date Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .cutsom-post-block-list-inner .post-date-box.post-block-meta-date" => "color: {{VALUE}}"
            ]
        ]);
		
		$this->add_control('postlist_small_date_color', [
            'label' => esc_html__('Small Post Date Color', 'vibenews-extra'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .cutsom-post-block-list-inner .post-block-list-small-inner .post-date-box.post-block-meta-date" => "color: {{VALUE}}"
            ]
        ]);
		
		
		
		$this->add_control(
          'title_length',
          [
            'label'         => esc_html__( 'Post Title Length', 'vibenews-extra' ),
            'type'          => \Elementor\Controls_Manager::NUMBER,
            'default'       => '10',
          ]
        );
		
		$this->add_control(
          'title_length_small',
          [
            'label'         => esc_html__( 'Small Post Title Length', 'vibenews-extra' ),
            'type'          => \Elementor\Controls_Manager::NUMBER,
            'default'       => '10',
          ]
        );

        $this->add_control(
          'content_length',
          [
            'label'         => esc_html__( 'Post Excerpt Length', 'ennlil-extra' ),
            'type'          => \Elementor\Controls_Manager::NUMBER,
            'default'       => '20',
          ]
        );
		
		$this->add_control(
          'content_length_small',
          [
            'label'         => esc_html__( 'Small Post Excerpt Length', 'ennlil-extra' ),
            'type'          => \Elementor\Controls_Manager::NUMBER,
            'default'       => '20',
          ]
        );
		

		
		$this->end_controls_section();
	
	}	
		


	protected function render() {
		
		
		$settings = $this->get_settings_for_display();

		$title_limit = $settings['title_length'];
		$title_limit_small = $settings['title_length_small'];
		
		
		$content_limit = $settings['content_length'];
		
		$content_limit_small = $settings['content_length_small'];
		
		
		$post_count = $settings['post_number'];
		$post_order  = $settings['post_ordering'];
		$post_sortby = $settings['post_sorting']; 
		
		$display_blog_excerpt = $settings['display_excerpt'];
		$display_blog_excerpt_small = $settings['display_excerpt_small'];
		

		$display_blog_author = $settings['display_author'];
		
		$display_blog_date = $settings['display_date'];
		$display_blog_date_small = $settings['display_date_small'];
		
		
		$display_blog_comment = $settings['blog_comment'];
		$display_read_time = $settings['display_read_time'];
		
		$display_top_title = $settings['post_list_title_enable'];
		
		$string_ID = $settings['post_ids'];
        $post_ID = ( ! empty( $string_ID ) ) ? array_map( 'intval', explode( ',', $string_ID ) ) : '';

		
		$args = [
            'post_type' => 'post',
            'post_status' => 'publish',
			'order' => $settings['post_ordering'],
			'posts_per_page' => $settings['post_number'],
			'tag__in'        => $settings['post_tags'],
			'ignore_sticky_posts' => 1,
			//'suppress_filters' => false,
        ];
		
		// Category
        if ( ! empty( $settings['post_categories'] ) ) {
            $args['category_name'] = implode(',', $settings['post_categories']);
        }
		
		// Post Sorting
        if ( ! empty( $settings['post_sorting'] ) ) {
            $args['orderby'] = $settings['post_sorting'];
        }	
		
		// Post Offset		
		if($settings['enable_offset_post'] == 'yes') {
			$args['offset'] = $settings['post_offset_count'];
		}

		//Specific Posts by ID's
        if ( ! empty( $settings['post_ids'] ) ) {
            $args['post__in'] = $post_ID;
        }



		// Query
        $query = new \WP_Query( $args ); ?>
		
		<section class="cutsom-post-block-list-wrapper">
			
			<?php if($display_top_title=='yes'): ?>	
			<div class="post-block-title-wrapper">
				<h2 class="post-block-title"><span><?php echo wp_kses_post( $settings['post_list_sec_title']  ); ?></span></h2>
			</div>
			<?php endif; ?>
		
		
			<?php if ( $query->have_posts() ) : ?>

			<div class="cutsom-post-block-list-inner">
				
				<?php 
			
				$i = 1;
				$count = 0;
				

				while ( $query->have_posts() ) :
					$query->the_post();


				$count++;

				  if ( $i == 1 || $i < 2 ) {
						$this->largepostcolumns( get_the_ID() );
					} else {
						$this->smallpostcolumns( get_the_ID() );

					}

					$i++;

				endwhile; ?>

			</div>
				
			<?php endif; ?>

			<?php wp_reset_postdata(); ?>

        </section>
		
<?php 

}
    protected function largepostcolumns() { ?>
			
			
            <article <?php post_class( 'post-list-top' ); ?>>
                <div class="blog-post-grid-wrapper post-block-list-wrap">
					<div class="news-post-grid-thumbnail">
						<a href="<?php the_permalink(); ?>" class="news-post-grid-thumbnail-wrap">
							<img src="<?php echo esc_attr(esc_url(get_the_post_thumbnail_url(null, 'full'))); ?>" alt="<?php the_title_attribute(); ?>">
						</a>
					</div>
					
					<div class="news-post-grid-content grid-content-inner">
						<div class="blog-post-meta-items post-block-meta-top">

							<div class="post-cat-box post-block-meta-cat">
							<?php require VIBENEWS_THEME_DIR . '/template-parts/cat-color.php'; ?>
							</div>
	
							
							<div class="post-date-box post-block-meta-date">
							<?php echo esc_html( get_the_date( 'F j, Y' ) ); ?>
							</div>
							
						</div>
						
						<h3 class="post-title">
							<a href="<?php the_permalink(); ?>"><?php echo esc_html( wp_trim_words(get_the_title(), 30,'') ); ?></a>
						</h3>
					

					</div>
				</div>
            </article>

        <?php
    }

    protected function smallpostcolumns() { ?>

			
			<article <?php post_class( 'post-list-bottom' ); ?>>
                <div class="blog-post-grid-wrapper post-block-list-small-wrap">
					<div class="news-post-grid-thumbnail">
						<a href="<?php the_permalink(); ?>" class="news-post-grid-thumbnail-wrap">
							<img src="<?php echo esc_attr(esc_url(get_the_post_thumbnail_url(null, 'full'))); ?>" alt="<?php the_title_attribute(); ?>">
						</a>
					</div>
					
					<div class="news-post-grid-content grid-content-inner post-block-list-small-inner">
						<div class="blog-post-meta-items post-block-meta-top">

							<div class="post-cat-box post-block-meta-cat">
							<?php require VIBENEWS_THEME_DIR . '/template-parts/cat-color.php'; ?>
							</div>

							
							<div class="post-date-box post-block-meta-date">
							<?php echo esc_html( get_the_date( 'F j, Y' ) ); ?>
							</div>
							
						</div>
						
						<h3 class="post-title">
							<a href="<?php the_permalink(); ?>"><?php echo esc_html( wp_trim_words(get_the_title(), 30,'') ); ?></a>
						</h3>
					

					</div>
				</div>
            </article>
              


 

        <?php

    }


   
   	public function posts_cat_list() {
		
		$terms = get_terms( array(
			'taxonomy'    => 'category',
			'hide_empty'  => true
		) );

		$cat_list = [];
		foreach($terms as $post) {
		$cat_list[$post->slug]  = [$post->name];
		}
		return $cat_list;
	  
	}		
	
}

